/***************************************************************************
 *
 * Copyright 2010,2011 BMW Car IT GmbH
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include "Layermanager.h"
#include "Log.h"
#include <iomanip>
#include "CommandList.h"
#include "LayerList.h"
#include "ICommand.h"
#include "IHealthMonitor.h"
#include "ICommunicator.h"
#include "IRenderer.h"
#include "ISceneProvider.h"
#include "Scene.h"
#include "Configuration.h"
#include "PluginManager.h"
#include "InputDeviceConfiguration.h"

#include <unistd.h>
#include <pthread.h>
#include <signal.h>
#include <limits.h>

#include <dirent.h>
#include <errno.h>
#include <dlfcn.h>
#include <stdlib.h>
#include "vncserver_common.h"

/* VNC plugin interface struct */
static struct vncserver_plugin_interface *gpst_plugin_interface = NULL;
static const char* NO_SENDER_NAME = "unknown";

Layermanager::Layermanager(Configuration& config)
:mConfiguration(config)
{
    m_pScene = new Scene();
    m_pPluginManager = new PluginManager(*this, mConfiguration);
    m_pApplicationReferenceMap = new ApplicationReferenceMap();

    m_pRendererList = new RendererList();
    m_pCommunicatorList = new CommunicatorList();
    m_pSceneProviderList = new SceneProviderList();
    m_pHealthMonitorList = new HealthMonitorList();
    m_pConfiguratorList = new ConfiguratorList();

    m_pPluginManager->getRendererList(*m_pRendererList);
    m_pPluginManager->getCommunicatorList(*m_pCommunicatorList);
    m_pPluginManager->getSceneProviderList(*m_pSceneProviderList);
    m_pPluginManager->getHealthMonitorList(*m_pHealthMonitorList);
    m_pPluginManager->getConfiguratorList(*m_pConfiguratorList);

    readInputDeviceConfigs(m_inputDeviceConfigurations);

    mHealthState = true;
    renderer_cnt = 0;
    communicator_cnt = 0;
    sceneprovider_cnt = 0;
    configurator_cnt = 0;
    m_vncserverInitialised = 0;
}

Layermanager::~Layermanager()
{
    if (m_pRendererList)
    {
        delete m_pRendererList;
        m_pRendererList = NULL;
    }

    if (m_pCommunicatorList)
    {
        delete m_pCommunicatorList;
        m_pCommunicatorList = NULL;
    }

    if (m_pSceneProviderList)
    {
        delete m_pSceneProviderList;
        m_pSceneProviderList = NULL;
    }

    if (m_pHealthMonitorList)
    {
        delete m_pHealthMonitorList;
        m_pHealthMonitorList = NULL;
    }

    if (m_pConfiguratorList)
    {
        delete m_pConfiguratorList;
        m_pConfiguratorList = NULL;
    }

    if (m_pScene)
    {
        delete m_pScene;
        m_pScene = NULL;
    }

    if (m_pApplicationReferenceMap)
    {
        ApplicationReferenceMap::iterator iter = m_pApplicationReferenceMap->begin();
        ApplicationReferenceMap::iterator end = m_pApplicationReferenceMap->end();
        for (; iter != end; ++iter)
        {
            delete iter->second;
        }
        delete m_pApplicationReferenceMap;
        m_pApplicationReferenceMap = NULL;
    }

    if (m_pPluginManager)
    {
        delete m_pPluginManager;
        m_pPluginManager = NULL;
    }

    if (!m_inputDeviceConfigurations.empty())
    {
        LayerManagerCalibration::InputDeviceConfigurationMap::iterator it;
        for (it = m_inputDeviceConfigurations.begin(); it != m_inputDeviceConfigurations.end(); ++it)
            delete it->second;
    }
}

uint Layermanager::getLayerTypeCapabilities(const LayerType layertype) const
{
    uint capabilities = 0;

    // get the first renderer for now TODO
    // change to possibly load multiple renderers

    IRenderer* renderer = *m_pRendererList->begin();
    if (renderer)
    {
        capabilities = renderer->getLayerTypeCapabilities(layertype);
    }

    return capabilities;
}

uint Layermanager::getNumberOfHardwareLayers(const uint screenID) const
{
    uint numberOfHardwareLayers = 0;

    // get the first renderer for now TODO
    // change to possibly load multiple renderers
    IRenderer* renderer = *m_pRendererList->begin();

    if (renderer)
    {
        numberOfHardwareLayers = renderer->getNumberOfHardwareLayers(screenID);
    }

    return numberOfHardwareLayers;
}

uint* Layermanager::getScreenResolution(const uint screenID) const
{
    uint* returnValue = 0;

    // get the first renderer for now TODO
    // change to possibly load multiple renderers
    IRenderer* renderer = *m_pRendererList->begin();

    if (renderer)
    {
        returnValue = renderer->getScreenResolution(screenID);
    }

    return returnValue;
}

float Layermanager::getScreenFPS(const uint screenID) const
{
    float returnValue = 0.0f;

    // get the first renderer for now TODO
    // change to possibly load multiple renderers
    IRenderer* renderer = *m_pRendererList->begin();

    if (renderer)
    {
        returnValue = renderer->getScreenFPS(screenID);
    }

    return returnValue;
}

uint* Layermanager::getScreenIDs(uint* length) const
{
    uint* returnValue = 0;

    // get the first renderer for now TODO
    // change to possibly load multiple renderers
    IRenderer* renderer = *m_pRendererList->begin();
    if (renderer)
    {
        returnValue = renderer->getScreenIDs(length);
    }

    return returnValue;
}

// This function is used to propagate a signal to the window system
// via the renderer for a redraw because of a rendering property change
void Layermanager::signalRendererRedraw()
{
    IRenderer* renderer = *m_pRendererList->begin();
    if (renderer)
    {
        renderer->signalWindowSystemRedraw();
    }
}

void Layermanager::addApplicationReference(t_ilm_client_handle client, IApplicationReference* reference)
{
    if (client && reference)
    {
        m_pidToProcessNameTable[reference->getProcessId()] = reference->getProcessName();
        (*m_pApplicationReferenceMap)[client]=reference;
        unsigned int pid = getSenderPid(client);
        LOG_INFO("Layermanager",
                 "Connect from application " << getSenderName(client)
                 << "(" << pid << ")");
    }
}

void Layermanager::removeApplicationReference(t_ilm_client_handle client)
{
    if (client)
    {
        unsigned int pid = getSenderPid(client);
        LOG_INFO("Layermanager",
                 "Disconnect from application " << getSenderName(client)
                 << "(" << pid << ")");
        delete (*m_pApplicationReferenceMap)[client];
        m_pApplicationReferenceMap->erase(client);
        m_pidToProcessNameTable.erase(pid);

        // if pid = 0, the client did not send ServiceConnect.
        // since many client may use this configuration, we must
        // not discard any pending commands here: we could discard
        // commands of currently running applications
        if (0 != pid)
        {
            CommandList& commandList = m_EnqueuedCommands[pid];
            CommandList::iterator iter = commandList.begin();
            CommandList::iterator end = commandList.end();
            for (; iter != end; ++iter)
            {
                delete *iter;
            }

            m_EnqueuedCommands[pid].clear();
            m_EnqueuedCommands.erase(pid);
        }
    }
}

t_ilm_uint Layermanager::getSenderPid(t_ilm_client_handle client)
{
    t_ilm_uint result = 0;
    IApplicationReference* app = (*m_pApplicationReferenceMap)[client];
    if (app)
    {
        result = app->getProcessId();
    }
    return result;
}

const char*  Layermanager::getSenderName(t_ilm_client_handle client)
{
    const char* result = NO_SENDER_NAME;
    IApplicationReference* app = (*m_pApplicationReferenceMap)[client];
    if (app)
    {
        result = app->getProcessName();
    }
    return result;
}

const char* Layermanager::getSenderName(unsigned int pid)
{
    const char* result = m_pidToProcessNameTable[pid];
    if (!result)
    {
        result = NO_SENDER_NAME;
    }
    return result;
}


void Layermanager::printDebugInformation() const
{
    // print stuff about layerlist
    LOG_INFO("Layermanager",
             "Layer: ID |  X |  Y |  W |  H |Al.| Z \n");

    unsigned int length;
    unsigned int* LayerIDs;

    // loop the layers
    m_pScene->getLayerIDs(&length, &LayerIDs);

    m_pScene->lockScene();

    for (unsigned int i = 0; i < length; i++)
    {
        Layer* currentLayer = m_pScene->getLayer(LayerIDs[i]);
        LOG_INFO("Layermanager",
                 "      " << std::setw(4) << currentLayer->getID() << "\n");

        LOG_INFO("Layermanager",
                 "    Surface:  ID |Al.| Z |  SVP: X |  Y |  W |  H     "
                 "DVP:  X |  Y |  W |  H \n");

        // loop the surfaces of within each layer
        SurfaceListConstIterator iter = currentLayer->getAllSurfaces().begin();
        SurfaceListConstIterator iterEnd = currentLayer->getAllSurfaces().end();

        for (; iter != iterEnd; ++iter)
        {
            LOG_INFO("Layermanager",
                     "            " << std::setw(4) << (*iter)->getID()
                     << std::setw(4) << std::setprecision(3)
                     << (*iter)->opacity << "\n");
        }
    }
    m_pScene->unlockScene();

    delete [] LayerIDs;
}

bool Layermanager::executeCommand(ICommand* commandToBeExecuted)
{
    ExecutionResult status = ExecutionFailed;
    m_pScene->lockScene();
    status = commandToBeExecuted->execute(this);
    m_pScene->unlockScene();

    if (status == ExecutionFailedRedraw || status == ExecutionSuccessRedraw)
    {
        // either a command that has changed the output was executed, or
        // commit command was executed which contained commands that requested redraw was executed.

        m_pScene->lockScene();
        signalRendererRedraw();
        m_pScene->unlockScene();
    }

    unsigned int commandPid = commandToBeExecuted->getSenderPid();
    if (status == ExecutionSuccess || status == ExecutionSuccessRedraw)
    {
        LOG_INFO("Layermanager",
             "Command=" << commandToBeExecuted->getString()
             << " from " << getSenderName(commandPid)
             << "(" << commandPid << ")"
             << " ,success+");
    }
    else
    {
        LOG_WARNING("Layermanager",
             "Command=" << commandToBeExecuted->getString()
             << " from " << getSenderName(commandPid)
             << "(" << commandPid << ")"
             << ", failed-");
    }

    return (status == ExecutionSuccess || status == ExecutionSuccessRedraw);
}

bool Layermanager::enqueueCommand(ICommand* command)
{
    bool result = false;

    if (command)
    {
        const unsigned int commandPid = command->getSenderPid();
        m_EnqueuedCommands[commandPid].push_back(command);
        result = true;

        LOG_DEBUG("Layermanager",
                    "Command=" << command->getString()
                    << " from " << getSenderName(commandPid)
                    << "(" << commandPid << ")"
                    << (result ? "+" : "-"));
    }

    return result;
}

bool Layermanager::execute(ICommand* commandToBeExecuted)
{
    bool status = false;

    ExecutionType type = commandToBeExecuted->getExecutionType();

    switch (type)
    {
    case ExecuteSynchronous:
        status = executeCommand(commandToBeExecuted);
        delete commandToBeExecuted;
        break;

    case ExecuteAsynchronous:
        status = enqueueCommand(commandToBeExecuted);
        break;

    default:
        // TODO: error
        LOG_WARNING("Layermanager",
                    "Command=" << commandToBeExecuted->getString()
                    << " was not processed");
        break;
    }

    return status;
}

int Layermanager::getPluginReportIntervalInMs()
{
    int retVal = INT_MAX;
    HealthMonitorList monitors;
    m_pPluginManager->getHealthMonitorList(monitors);
    
    if (0 == monitors.size())
    {
        retVal = -1;
    }
    else
    {
        HealthMonitorListIterator iter = monitors.begin();
        HealthMonitorListIterator end = monitors.end();
        for (; iter != end; ++iter)
        {
            IHealthMonitor* monitor = *iter;
            int monitorInterval = monitor->getPluginReportIntervalInMs();
            retVal = (monitorInterval < retVal ? monitorInterval : retVal);
        }    
    }
    return retVal;
}

bool Layermanager::startAllRenderers(const int width, const int height,
        const char *displayName)
{
    bool allStarted = false;

    int maxIterationDurationInMS = getPluginReportIntervalInMs();

    RendererListIterator iter = m_pRendererList->begin();
    RendererListIterator iterEnd = m_pRendererList->end();
    for (; iter != iterEnd; ++iter)
    {
        IRenderer* renderer = *iter;
        if (renderer)
        {
            allStarted = renderer->start(width, height, displayName, maxIterationDurationInMS);
            if (allStarted)
            {
                renderer_cnt++;
                break;
            }
            else
            {
                allStarted = false;
                LOG_ERROR("Layermanager",
                        "Could not start Rendering Plugin");
            }
        }
    }

    return allStarted;
}

bool Layermanager::delegateScene()
{
    bool allStarted = true;

    SceneProviderListIterator iter = m_pSceneProviderList->begin();
    SceneProviderListIterator iterEnd = m_pSceneProviderList->end();

    for (; iter != iterEnd; ++iter)
    {
        ISceneProvider* sceneProvider = *iter;
        if (sceneProvider)
        {
            allStarted &= sceneProvider->delegateScene();
            if (allStarted)
            {
                sceneprovider_cnt++;
            }
            else
            {
                allStarted = false;
                LOG_WARNING("Layermanager",
                            "Could not start Scene Plugin, initial Scene is lost");
            }
        }
    }

    return allStarted;
}

bool Layermanager::startAllCommunicators()
{
    bool allStarted = true;

    int maxIterationDurationInMS = getPluginReportIntervalInMs();

    CommunicatorListIterator communicatorIter = m_pCommunicatorList->begin();
    CommunicatorListIterator communicatorIterEnd = m_pCommunicatorList->end();

    for (; communicatorIter != communicatorIterEnd; ++communicatorIter)
    {

        ICommunicator *communicator = *communicatorIter;
        if (communicator)
        {
            LOG_DEBUG("Layermanager",
                           "try to start Communication Controllers ");
            allStarted &= communicator->start(maxIterationDurationInMS);
            if (allStarted)
            {
                LOG_DEBUG("Layermanager",
                              "Communication Controllers started");
                communicator_cnt++;
            }
            else
            {
                allStarted = false;
                LOG_ERROR("Layermanager",
                          "Could not start Communication Controllers");
            }
        }
    }

    return allStarted;
}

bool Layermanager::startAllHealthMonitors()
{
    bool allStarted = true;

    HealthMonitorListIterator iter = m_pHealthMonitorList->begin();
    HealthMonitorListIterator iterEnd = m_pHealthMonitorList->end();

    for (; iter != iterEnd; ++iter)
    {
        IHealthMonitor* healthMonitor = *iter;
        if (healthMonitor)
        {
            allStarted &= healthMonitor->start();
        }
        else
        {
            allStarted = false;
        }
    }
    if (!allStarted)
    {
        LOG_ERROR("Layermanager", "Could not start Health Monitors");
    }
    return allStarted;
}

bool Layermanager::stopVncServer()
{
    bool ret_val = false;
    if (m_vncserverInitialised)
    {
        gpst_plugin_interface->deinit();
        gpst_plugin_interface->stop();
        m_vncserverInitialised = false;
        free(gpst_plugin_interface);
        gpst_plugin_interface   =    NULL;
        ret_val = true;
    }
    return ret_val;
}

int Layermanager::dumpFramebuffertovncsurf(int displayID,void *userdata)
{
    bool     returnValue = false;

    LOG_DEBUG("Layermanager", "called ");

    Layermanager* layermgr = static_cast<Layermanager*>((Layermanager*)userdata);

    layermgr->m_pScene->lockScene();

    IRenderer* renderer = *(layermgr->m_pRendererList)->begin();
    if (renderer)
    {
        returnValue = renderer->dumpFramebuffertovncsurf(displayID);
    }
    layermgr->m_pScene->unlockScene();

     return returnValue;

}

int Layermanager::getVncDisplayConnectionState(int displayID,
                                                                        void **pp_vncbuffaddr,void **pp_vncsurfptr)
{
    int ret_val = false;
    LOG_DEBUG("Layermanager", "called ");

    if (m_vncserverInitialised)
    {
        ret_val = gpst_plugin_interface->getVNCdisplayConnectionstate(displayID,pp_vncbuffaddr,
                                                                                                pp_vncsurfptr);
    }
    return ret_val;
}

int Layermanager::NotifyVnc_BeforeFramebufferUpdate(int displayID)
{
    int ret_val = false;
    LOG_DEBUG("Layermanager", "called ");

    if (m_vncserverInitialised)
    {
        ret_val = gpst_plugin_interface->notifyvncbufferupatebefore(displayID);
    }
    return ret_val;
}

int Layermanager::NotifyVnc_FramebufferUpdated(int displayID)
{
    int ret_val = false;
    LOG_DEBUG("Layermanager", "called ");

    if (m_vncserverInitialised)
    {
        ret_val = gpst_plugin_interface->notifyvncbufferupated(displayID);
    }
    return ret_val;
}

int Layermanager::destoryVncBuffer(int displayID,void* p_vncsurfptr, void *userdata)
{
    bool     returnValue = false;

    LOG_DEBUG("Layermanager", "called ");

    Layermanager* layermgr = static_cast<Layermanager*>((Layermanager*)userdata);
    IRenderer* renderer = *(layermgr->m_pRendererList)->begin();

    layermgr->m_pScene->lockScene();
    if (renderer)
    {
        returnValue = renderer->destroyVncbuffer(displayID,p_vncsurfptr);
    }
    layermgr->m_pScene->unlockScene();

    return returnValue;

}
int Layermanager::createVncBuffer(int displayID,int bpp,void** pp_vncbuffaddr,
                                                    void** pp_vncsurfptr,void* userdata)
{
    bool     returnValue = false;

    LOG_DEBUG("Layermanager", "called ");

    Layermanager* layermgr = static_cast<Layermanager*>((Layermanager*)userdata);
    IRenderer* renderer = *(layermgr->m_pRendererList)->begin();

    layermgr->m_pScene->lockScene();
    if (renderer)
    {
        returnValue = renderer->createVncbuffer(displayID,bpp,pp_vncsurfptr,pp_vncbuffaddr);
    }
    layermgr->m_pScene->unlockScene();

     return returnValue;
}

int Layermanager::processVncEvents(int displayID,void *pvncevent,void *userdata)
{
    bool     returnValue = false;
    vncevent *p_vncevent = (vncevent*)pvncevent;
    LOG_DEBUG("Layermanager", "called ");

    Layermanager* layermgr = static_cast<Layermanager*>((Layermanager*)userdata);
    IRenderer* renderer = *(layermgr->m_pRendererList)->begin();

    layermgr->m_pScene->lockScene();

    switch(p_vncevent->eventtype)
    {
        /*
          * Currently generates the touch event from vnc mouse events
          * VNC touch events is not supported once vncserver supports touch event, 
          * will call separate api to process.
          */
        case VNC_TOUCH_EVENT:
        {
            if (renderer)
            {
                returnValue = renderer->processVnctouchevent(displayID, 
                                                                        p_vncevent->x,
                                                                        p_vncevent->y,
                                                                        p_vncevent->down);
            }
        }
         break;

        case VNC_MOUSE_EVENT:
        {
            if (renderer)
            {
                returnValue = renderer->processVnctouchevent(displayID, 
                                                                        p_vncevent->x,
                                                                        p_vncevent->y,
                                                                        p_vncevent->down);
                /*
                  * Currently generates the touch event from vnc mouse events since
                  * vncserver is not supported touch events
                  */
                  /* returnValue = renderer->processVncmouseevent(displayID,
                                                                                    p_vncevent->x,
                                                                                    p_vncevent->y,
                                                                                    p_vncevent->buttonmask);
                */
            }
        }
        break;

        case VNC_KBD_EVENT:
        {
            if (renderer)
            {
                returnValue = renderer->processVnckeyboardevent(displayID, 
                                                                        p_vncevent->keycode,
                                                                        p_vncevent->down);
            }
        }
        break;
    } 
    layermgr->m_pScene->unlockScene(); 

    return returnValue;
}


bool Layermanager::initialiseVncServer(void *p_plugininterface)
{
    vncserver_info  stvncserverinfo = { 0 };
    displayinfo     *pst_dispinfo = NULL;
    uint     length = 0;
    uint     loop = 0;
    uint* screenIDs = getScreenIDs(&length);
    uint width = 0;
    uint height = 0;
    uint* p_screenresolution;
    char a_displayname[20] = { 0 };
    bool    ret_val = false;

    plugin_interface_info *pst_plugininterface = (plugin_interface_info*)p_plugininterface;

    LOG_DEBUG("Layermanager", " called ");

    strcpy(a_displayname,"display");

    if( length )
    {
        pst_dispinfo =  (displayinfo*)malloc(length*sizeof(displayinfo));
        memset(pst_dispinfo,0,(length*sizeof(displayinfo)) );

        if(NULL != pst_dispinfo)
        {
            while(loop < length)
            {
                width = 0;
                height = 0;
                p_screenresolution = getScreenResolution(screenIDs[loop]);
                LOG_DEBUG("Layermanager", " length= "<<length);

                if(NULL != p_screenresolution)
                {
                    if(p_screenresolution[0] != -1)
                    {
                        width = p_screenresolution[0];
                    }
                    if(p_screenresolution[1] != -1)
                    {
                        height = p_screenresolution[1];
                    }
                    delete [] p_screenresolution;
                    p_screenresolution = NULL;
                }
                pst_dispinfo[loop].width  = width;
                pst_dispinfo[loop].height = height;
                pst_dispinfo[loop].displayID = screenIDs[loop];
                sprintf(a_displayname, "display-%i", screenIDs[loop]);
                memcpy( pst_dispinfo[loop].a_displayname,a_displayname,
                                ( sizeof(pst_dispinfo[loop].a_displayname) -1) );
                LOG_DEBUG("Layermanager", "width="<<pst_dispinfo[loop].width<<" ,"
                                    "height="<<pst_dispinfo[loop].height<<" ,"
                                    "displayID="<<pst_dispinfo[loop].displayID<<" ,"
                                    "a_displayname="<<pst_dispinfo[loop].a_displayname );


                loop++;
            }
            stvncserverinfo.p_displayinfo = pst_dispinfo;
            stvncserverinfo.noofdisplays    =   length;
            stvncserverinfo.userdata        =   (void*)this;

            /* initialise the plugin */
            ret_val = pst_plugininterface->st_plugin_interface.init(&stvncserverinfo);

            /* Free the allocated local displayinfo memory */
            free(pst_dispinfo);
            delete [] screenIDs;
        }
    }
    return ret_val;
}

bool Layermanager::loadVncServerSymbols(void *libraryHandle,void *p_plugininterface)
{
    const char* dlopen_error;
    bool ret_val = true;
    plugin_interface_info *pst_plugininterface = (plugin_interface_info*)p_plugininterface;

    LOG_DEBUG("Layermanager", "called ");

    pfnentrypoint    entrypoint    = (pfnentrypoint) dlsym(libraryHandle,LIB_VNCSERVER_PLUGIN_ENTRY_POINT );

    dlopen_error= dlerror();
    if (dlopen_error)
    {
        LOG_ERROR("Layermanager", "can't find "<<LIB_VNCSERVER_PLUGIN_ENTRY_POINT<<" function");
        return false;
    }

    /*
      * Assiging call back functions which will be called from vncserver plugin
      */
    pst_plugininterface->st_plugin_clkbk.createvncbuffer  =   createVncBuffer;
    pst_plugininterface->st_plugin_clkbk.processevents    =   processVncEvents;
    pst_plugininterface->st_plugin_clkbk.dumpframebuffertovncsurf =\
                                                                dumpFramebuffertovncsurf;
    pst_plugininterface->st_plugin_clkbk.destoryvncbuffer = \
                                                                    destoryVncBuffer;

    ret_val = entrypoint(pst_plugininterface);

    if ( ret_val )
    {
        /* copy the plugin interfaces */
        memcpy(gpst_plugin_interface,&(pst_plugininterface->st_plugin_interface),
                        sizeof(struct vncserver_plugin_interface) );
    }

    LOG_DEBUG("Layermanager", "ret_val= "<<ret_val);
    return ret_val;
}

bool Layermanager::startVncServer()
{
    // open library
    void *libraryHandle;
    char*   path= (char*) LIB_VNCSERVER_PLUGIN_PATH;
    const char* dlopen_error;
    struct plugininterfaceinfo st_plugininterface = { 0 };
    bool val  = false;
    bool allStarted = true;

    dlerror(); // Clear any existing error
    libraryHandle = dlopen(path, RTLD_NOW | RTLD_GLOBAL);
    dlopen_error= dlerror();
    if (dlopen_error)
    {
        LOG_WARNING("Layermanager", "dlopen_error="<<dlopen_error);
        dlerror(); // Clear any existing error
        return allStarted;
    }

    LOG_DEBUG("Layermanager", "vncserver plugin-"<<LIB_VNCSERVER_PLUGIN_PATH
                        <<" loaded");

    gpst_plugin_interface   =  (struct vncserver_plugin_interface*) \
                            malloc(sizeof(struct vncserver_plugin_interface));

    if(NULL == gpst_plugin_interface)
    {
        LOG_DEBUG("Layermanager", "gpst_plugin_interface cant be allocated");
        return false;
    }

    memset(gpst_plugin_interface,0,(sizeof(struct vncserver_plugin_interface)) );

    val = loadVncServerSymbols(libraryHandle,&st_plugininterface);
    if( false == val )
    {
        LOG_ERROR("Layermanager", "vncserver plugin symbols not loaded-error");
        return val;
    }

    val = initialiseVncServer((void*)&st_plugininterface);
    if( true == val )
    {
        /* call to run the vncserver */
        val = gpst_plugin_interface->start();
        if( true == val )
        {
            m_vncserverInitialised = true;
        }
        else
        {
            LOG_ERROR("Layermanager", "start- vncserver  failed");
            allStarted  = false;
        }
    }
    else
    {
        LOG_ERROR("Layermanager", "initialiseVncServer failed");
        allStarted  = false;
    }
    
    LOG_DEBUG("Layermanager", "ret_val= "<<allStarted);
    return allStarted;
}

bool Layermanager::startManagement()
{
    bool result = false;

    const int width = mConfiguration.getDisplayWidth();
    const int height = mConfiguration.getDisplayHeight();
    const char* displayName = mConfiguration.getDisplayName().c_str();

    // 1. start renderers, no prerequisites
    // 2. execute scene provider plugins
    // 3. start communication (after scene is ready to use)
    result = runtimeConfiguration()
                && startAllRenderers(width, height, displayName)
                && delegateScene()
                && startAllCommunicators()
                && startAllHealthMonitors()
                && startVncServer();

    return result;
}

void Layermanager::stopAllHealthMonitors()
{
    HealthMonitorListIterator iter = m_pHealthMonitorList->begin();
    HealthMonitorListIterator iterEnd = m_pHealthMonitorList->end();
    for (; iter != iterEnd; ++iter)
    {
        IHealthMonitor* healthMonitor = *iter;
        if (healthMonitor)
        {
            healthMonitor->stop();
        }
    }
}

void Layermanager::stopAllRenderers()
{
    RendererListIterator iter = m_pRendererList->begin();
    RendererListIterator iterEnd = m_pRendererList->end();
    for (; iter != iterEnd; ++iter)
    {
        IRenderer *renderer = *iter;
        if (renderer)
        {
            renderer->stop();
        }
    }
}

void Layermanager::stopAllCommunicators()
{
    CommunicatorListIterator iter = m_pCommunicatorList->begin();
    CommunicatorListIterator iterEnd = m_pCommunicatorList->end();
    for (; iter != iterEnd; ++iter)
    {
        ICommunicator *communicator = *iter;
        if (communicator)
        {
            communicator->stop();
        }
    }
}

bool Layermanager::stopManagement()
{
    stopVncServer();
    stopAllHealthMonitors();
    stopAllRenderers();
    stopAllCommunicators();
    return true; // TODO
}

HealthCondition Layermanager::getHealth()
{
    HealthCondition returnValue = HealthRunning;
    PluginList::const_iterator iter = mMonitoredPlugins.begin();
    PluginList::const_iterator iterEnd = mMonitoredPlugins.end();

    while ((iter != iterEnd) && (HealthRunning == returnValue))
    {
        IPlugin* monitoredPlugin = *iter;
        returnValue = monitoredPlugin->pluginGetHealth();
        LOG_INFO("Layermanager",
                 "Plugin " << monitoredPlugin->pluginGetName()
                 << ", health: "
                 << (returnValue == HealthRunning ? "Running" : "Dead"));
        ++iter;
    }

    return returnValue;
}

void Layermanager::readInputDeviceConfigs(LayerManagerCalibration::InputDeviceConfigurationMap& map)
{
    const string default_config_file(this->mConfiguration.getInputDevConfigPath());

    LayerManagerCalibration::InputDeviceConfiguration* conf = new LayerManagerCalibration::InputDeviceConfiguration;
    if (conf->parseFile(default_config_file.c_str()))
    {
        LOG_INFO("Layermanager",
                 "Successfully read config file, name=" << default_config_file);
        const list<string*>& deviceNames = conf->getDeviceNames();
        list<string*>::const_iterator iter;

        for (iter = deviceNames.begin();
             iter != deviceNames.end();
             iter++)
        {
            map.insert(std::pair<std::string,LayerManagerCalibration::InputDeviceConfiguration*>(**iter, conf));
        }
    }
    else
    {
        LOG_WARNING("Layermanager",
                  "Failed to read config file, name=" << default_config_file);
    }
}

LayerManagerCalibration::InputDeviceConfigurationMap* Layermanager::getInputDeviceConfigurations()
{
    return &m_inputDeviceConfigurations;
}

void Layermanager::getResolution(uint* width, uint* height)
{
    *width = mConfiguration.getDisplayWidth();
    *height = mConfiguration.getDisplayHeight();
}

bool Layermanager::runtimeConfiguration()
{
    bool allStarted = true;

    ConfiguratorListIterator iter = m_pConfiguratorList->begin();
    ConfiguratorListIterator iterEnd = m_pConfiguratorList->end();

    for (; iter != iterEnd; ++iter)
    {
        IConfigurator* configurator = *iter;
        if (configurator)
        {
            allStarted &= configurator->configure();
            if (allStarted)
            {
                configurator_cnt++;
                /*parse device configurtation*/
                readInputDeviceConfigs(m_inputDeviceConfigurations);
            }
            else
            {
                allStarted = false;
                LOG_WARNING("Layermanager",
                            "Could not execute runtime configuration");
            }
        }
    }

    return allStarted;
}
